/*
 * Routines dealing with routing
 */

#include "libfma.h"
#include "lf_fabric.h"
#include "lf_fms_comm.h"
#include "lf_topo_map.h"
#include "libmyri.h"

#include "fma.h"
#include "fma_myri.h"
#include "fma_map.h"
#include "fma_fabric.h"
#include "fma_standalone.h"

/*
 * local prototypes
 */
static void fma_calc_quick_routes(struct fma_nic_info *nip, int port);
void print_fanout(struct fma_nic_info *nip, int port);
void dumpit(struct lf_fabric *fp);

void
show_routes()
{
  struct lf_fabric *fp;

  struct fma_nic_info *lcl_nic;   /* this host fma struct */
  int lcl_nic_id;                 /* this hosts nic id */
  int lcl_port;                    /* local nic interface */
  struct lf_nic *rmt_nic;         /* remote nic struct */
  int rmt_nic_id;                 /* remote nic id */
  int rmt_port;                    /* remote nic interface */
  unsigned char *route;
  int rlen;
  int np_index;
  int rmt_np_index;
  int n;
  int rindex;

  fp = A.fabric;

  for (rmt_nic_id = 0; rmt_nic_id < FMA_FABRIC(fp)->num_nics; ++rmt_nic_id) {

    rmt_nic = FMA_FABRIC(fp)->nics[rmt_nic_id];

    for (lcl_nic_id = 0; lcl_nic_id < A.myri->num_nics; ++lcl_nic_id) {

      lcl_nic = A.myri->nic_info[lcl_nic_id];

      for (lcl_port = 0; lcl_port < 2; lcl_port++) {

	np_index = FMA_NICPORT_INDEX(lcl_nic_id, lcl_port);
        for (rmt_port = 0; rmt_port < 2; rmt_port++) {
          rindex = FMA_ROUTE_INDEX(lcl_nic_id, lcl_port, rmt_port);

	  rmt_np_index = (np_index * LF_MAX_NIC_PORTS) + rmt_port;

	  for (n=0; n<FMA_NIC(rmt_nic)->dfs_route_count[rmt_np_index]; ++n) {

	    rlen = FMA_NIC(rmt_nic)->route_lens[rindex];
	    route = FMA_NIC(rmt_nic)->route_buf + FMA_ROUTE_OFFSET(rindex);

	    if (rlen != -1) {
	      fma_log("%d p%d -> " LF_MAC_FORMAT " p%d: [%d] %s len=%d",
		  lcl_nic_id, lcl_port,
		  LF_MAC_ARGS(rmt_nic->mac_addr), rmt_port, n,
		  lf_route_str(route, rlen), rlen);
	    }
	    ++rindex;
          }
        }
      }
    }
  }
}

/*
 * generate a really quick and dirty set of routes
 */
void
fma_calc_all_quick_routes()
{
  int n;
  int port;
  struct lf_fabric *fp;

  fp = A.fabric;

  for (n=0; n<fp->num_xbars; ++n) {
    FMA_XBAR(fp->xbars[n])->bf_serialp = NULL;
  }

  for (n=0; n<A.myri->num_nics; ++n) {
    struct fma_nic_info *nip;

    nip = A.myri->nic_info[n];
    
    /* calculate routes for each port on this NIC */
    for (port=0; port < nip->myri_info.num_ports; ++port) {
      fma_calc_quick_routes(nip, port);
    }
  }
}

/*
 * A debugging aid to show the fanout from a NIC
 */
void
print_fanout(
  struct fma_nic_info *nip,
  int port)
{
  struct lf_xbar *xp;
  struct lf_nic *nicp;
  enum { UP, DOWN, FLAT, NUM_DIRS };
  struct lf_xbar *work_list[NUM_DIRS];
  struct lf_xbar *new_work_list[NUM_DIRS];
  void *serialp;
  struct lf_fabric *fp;

  fp = A.fabric;

  printf("\nfanout:\n");
  /* find directly connected xbar */
  nicp = FMA_FABRIC(fp)->nics[nip->ni_map_index];
  xp = LF_XBAR(nicp->topo_ports[port]);
  if (xp == NULL) {
    printf("nothing connected\n");
    return;		/* just return if no xbar connected... */
  }

  /* set up route info for this xbar */
  FMA_XBAR(xp)->fx_in_port = nicp->topo_rports[port];
  FMA_XBAR(xp)->fx_route_len = 0;

  printf("direct: xbar at %p\n", xp);

  /* this is the tag we'll use for bfs serial no. */
  serialp = &serialp;

  /* initialize UP work_list to just be first xbar */
  new_work_list[UP] = xp;
  new_work_list[DOWN] = NULL;
  new_work_list[FLAT] = NULL;
  FMA_XBAR(xp)->bf_next = NULL;
  FMA_XBAR(xp)->bf_serialp = serialp;

  /* Breadth-first until nothing more to do */
  while (new_work_list[UP] != NULL
         || new_work_list[DOWN] != NULL
         || new_work_list[FLAT] != NULL) {
    int dir;

    /* Copy over work lists */
    for (dir=0; dir<NUM_DIRS; ++dir) {
      work_list[dir] = new_work_list[dir];
      new_work_list[dir] = NULL;
    }

    /* process all lists each time through */
    for (dir=0; dir<NUM_DIRS; ++dir) {

      /* save this list and start building new work_list */
      xp = work_list[dir];
      work_list[dir] = NULL;

      /* process every xbar in old work_list */
      while (xp != NULL) {
	int p;

	printf("\n");
	printf("xbar %p, level = %d\n", xp, xp->clos_level);

	/* Check all ports for NICs and unvisited xbars */
	for (p=0; p<xp->num_ports; ++p) {
	  union lf_node *np2;
	  int port2;

	  printf("%p:%d - ", xp, p);

	  /* if this link is down, skip it */
	  if (xp->link_state[p] != LF_LINK_STATE_UP) {
	    printf("DOWN\n");
	    continue;
	  }

	  np2 = xp->topo_ports[p];
	  port2 = xp->topo_rports[p];
	  if (np2 == NULL) {
	    printf("nothing\n");
	    continue;
	  }

	  /* If this is a NIC, save the route! */
	  if (np2->ln_type == LF_NODE_NIC) {

	    printf("NIC " LF_MAC_FORMAT "\n",
		LF_MAC_ARGS(LF_NIC(np2)->mac_addr));

	  } else if (np2->ln_type == LF_NODE_XBAR) {
	    struct lf_xbar *xp2;
	    int new_dir;

	    xp2 = LF_XBAR(np2);
	    printf("xbar %p:%d ", xp2, port2);

	    if (xp2->clos_level > xp->clos_level) {
	      printf("(UP) ");
	      new_dir = UP;
	    } else if (xp2->clos_level < xp->clos_level) {
	      printf("(DOWN) ");
	      new_dir = DOWN;
	    } else {
	      printf("(FLAT) ");
	      new_dir = FLAT;
	    }

	    if (FMA_XBAR_N(np2)->bf_serialp == serialp) {
	      printf("SEEN\n");
	      continue;
	    }

	    /* don't go from flat or down to anything but down! */
	    if (dir != UP && new_dir != DOWN) {
	      printf("wrongway\n");
	      continue;
	    }
	    printf("adding\n");

	    /* add to new work_list */
	    FMA_XBAR(xp2)->bf_next = new_work_list[new_dir];
	    new_work_list[new_dir] = xp2;
	    
	    /* mark as visited to prevent multiple visitors */
	    FMA_XBAR(xp2)->bf_serialp = serialp;
	  } else {
            printf("???\n");
	  }
	}

	/* next in list */
	xp = FMA_XBAR(xp)->bf_next;
      }
    }
  }
}

/*
 * Calculate a quick route to each host.
 *
 * start from directly connected xbar and breadth-first search
 * everywhere until we run out of places to go.  We go up then down
 * with one flat hop allowed.
 */

static void
fma_calc_quick_routes(
  struct fma_nic_info *nip,
  int port)
{
  int rlen;
  struct lf_host *hp;
  struct lf_xbar *xp;
  struct lf_nic *nicp;
  union lf_node *np;
  unsigned char *route;
  enum { UP, DOWN, FLAT, NUM_DIRS };
  struct lf_xbar *work_list[NUM_DIRS];
  struct lf_xbar *new_work_list[NUM_DIRS];
  void *serialp;
  struct lf_fabric *fp;

  fp = A.fabric;

  /* If NIC not in map, don't try to find it */
  if (nip->ni_map_index == -1) {
    return;
  }

  /* find directly connected xbar */
  nicp = FMA_FABRIC(fp)->nics[nip->ni_map_index];
  np = nicp->topo_ports[port];
  if (np == NULL) {
    return;		/* just return nothing connected */
  }

  /* If this is a NIC, set the 0-len route for it */
  if (np->ln_type == LF_NODE_NIC) {

    /* if quick_route_len is -1, we need a new quick route */
    hp = LF_NIC(np)->host;
    if (FMA_HOST(hp)->quick_route_len == -1) {
      FMA_HOST(hp)->quick_route_len = 0;
      FMA_HOST(hp)->quick_nic = nip;
      FMA_HOST(hp)->quick_port = port;
    }
    return;

  } else {
    xp = LF_XBAR(np);
  }

  /* set up route info for this xbar */
  FMA_XBAR(xp)->fx_in_port = nicp->topo_rports[port];
  FMA_XBAR(xp)->fx_route_len = 0;

  /* this is the tag we'll use for bfs serial no. */
  serialp = &(nicp->topo_ports[port]);

  /* initialize UP work_list to just be first xbar */
  new_work_list[UP] = xp;
  new_work_list[DOWN] = NULL;
  new_work_list[FLAT] = NULL;
  FMA_XBAR(xp)->bf_next = NULL;
  FMA_XBAR(xp)->bf_serialp = serialp;
  rlen = 1;

  /* Breadth-first until nothing more to do */
  while (new_work_list[UP] != NULL
         || new_work_list[DOWN] != NULL
         || new_work_list[FLAT] != NULL) {
    int dir;

    /* Copy over work lists */
    for (dir=0; dir<NUM_DIRS; ++dir) {
      work_list[dir] = new_work_list[dir];
      new_work_list[dir] = NULL;
    }

    /* process all lists each time through */
    for (dir=0; dir<NUM_DIRS; ++dir) {

      /* save this list and start building new work_list */
      xp = work_list[dir];
      work_list[dir] = NULL;

      /* process every xbar in old work_list */
      while (xp != NULL) {
	int p;
	int i;

	/* Check all ports for NICs and unvisited xbars */
	p = random() % xp->num_ports;
	for (i=0; i<xp->num_ports; ++i) {
	  union lf_node *np2;
	  int port2;

	  p = (p+1) % xp->num_ports;

	  /* don't violate quadrant disable */
	  if (lf_xbar_qd_violation(xp, FMA_XBAR(xp)->fx_in_port, p)) continue;

	  /* if this link is down, skip it */
	  if (xp->link_state[p] != LF_LINK_STATE_UP) continue;

	  np2 = xp->topo_ports[p];
	  port2 = xp->topo_rports[p];
	  if (np2 == NULL) continue;

	  /* If this is a NIC, save the route if needed */
	  if (np2->ln_type == LF_NODE_NIC) {

	    /* if quick_route_len is -1, we need a new quick route */
	    hp = LF_NIC(np2)->host;
	    if (FMA_HOST(hp)->quick_route_len == -1) {
	      route = FMA_HOST(hp)->quick_route;
	      memcpy(route, FMA_XBAR(xp)->fx_route, rlen-1);
	      route[rlen-1] = LF_DELTA_TO_ROUTE(p - FMA_XBAR(xp)->fx_in_port);
	      FMA_HOST(hp)->quick_route_len = rlen;
	      FMA_HOST(hp)->quick_nic = nip;
	      FMA_HOST(hp)->quick_port = port;
	    }

	    /* keep track of longest-shortest route between NIC ports */
	    if (rlen > A.map_info->mi_longest_route) {
	      A.map_info->mi_longest_route = rlen;
	    }

	  } else if (np2->ln_type == LF_NODE_XBAR
		     && FMA_XBAR_N(np2)->bf_serialp != serialp) {
	    struct lf_xbar *xp2;
	    int new_dir;

	    xp2 = LF_XBAR(np2);
	    if (xp2->clos_level > xp->clos_level) {
	      new_dir = UP;
	    } else if (xp2->clos_level < xp->clos_level) {
	      new_dir = DOWN;
	    } else {
	      new_dir = FLAT;
	    }

	    /* don't go from flat or down to anything but down! */
	    if (dir != UP && new_dir != DOWN) {
	      continue;
	    }

	    /* record route to this xbar */
	    memcpy(FMA_XBAR(xp2)->fx_route, FMA_XBAR(xp)->fx_route, rlen-1);
	    FMA_XBAR(xp2)->fx_route[rlen-1] =
		LF_DELTA_TO_ROUTE(p - FMA_XBAR(xp)->fx_in_port);
	    FMA_XBAR(xp2)->fx_route_len = rlen;
	    FMA_XBAR(xp2)->fx_in_port = port2;

	    /* add to new work_list */
	    FMA_XBAR(xp2)->bf_next = new_work_list[new_dir];
	    new_work_list[new_dir] = xp2;
	    
	    /* mark as visited to prevent multiple visitors */
	    FMA_XBAR(xp2)->bf_serialp = serialp;
	  }
	}

	/* next in list */
	xp = FMA_XBAR(xp)->bf_next;
      }
    }

    /* increment distance for each go-round */
    ++rlen;
  }

  /* show_routes(); */ /* XXX */
}

/*
 * Set routes from lcl NIC interfaces to rmt NIC interfaces. The routing between
 * the local and remote interfaces is described with 2 lcl and 2 rmt ports
 * and are iterated over for the number of routes to be set.
 *
 * If lcl_port_0 != lcl_port_1 and we are running an API that
 * accepts num_routes per port (e.g. MX) instead of per-NIC (e.g. GM),
 * then load num_routes to each of those routes.  If not equal and not per-port,
 * alternate one from each port until num_routes loaded.
 */
void
fma_set_route_group(
  struct fma_nic_info *lcl_nic,
  struct lf_nic *rmt_nic,
  int lcl_port_0,
  int rmt_port_0,
  int lcl_port_1,
  int rmt_port_1)
{
  int rt_num;
  int port_rt_num[2];
  int rc;
  int port_index;
  int lcl_ports[2];
  int rmt_ports[2];
  int lcl_port;
  int rmt_port;
  int rindex;
  int rlen;

  lcl_ports[0] = lcl_port_0;
  lcl_ports[1] = lcl_port_1;
  rmt_ports[0] = rmt_port_0;
  rmt_ports[1] = rmt_port_1;


  /* simplest case - all routes to and from same port */
  if (lcl_port_0 == lcl_port_1 && rmt_port_0 == rmt_port_1) {
    for (rt_num = 0; rt_num < lcl_nic->myri_info.num_routes; rt_num++) {

      rindex = FMA_ROUTE_INDEX(lcl_nic->nic_index, lcl_port_0, rmt_port_0)
		+ rt_num;

      rc = fma_set_route(lcl_nic, lcl_port_0, rmt_nic, rmt_port_0,
			 rindex, rt_num);
      if (rc == -1) {
	fma_log("Error setting route %d to %s", rt_num,
		fma_mac_to_hostname(rmt_nic->mac_addr));
      }
    }

  /*
   * If we will be switching port pairs, only increment NIC route index
   * when we switch port pairs
   */
  } else if (lcl_port_0 != lcl_port_1
             && lcl_nic->myri_info.num_routes_is_per_port) {

    for (rt_num = 0; rt_num < lcl_nic->myri_info.num_routes; rt_num++) {

      for (port_index=0; port_index<2; ++port_index) {
	lcl_port = lcl_ports[port_index];
	rmt_port = rmt_ports[port_index];

	rindex = FMA_ROUTE_INDEX(lcl_nic->nic_index, lcl_port, rmt_port)
	  + rt_num;

	rc = fma_set_route(lcl_nic, lcl_port, rmt_nic, rmt_port,
	    rindex, rt_num);
	if (rc == -1) {
	  fma_log("Error setting route %d to %s, p%d->p%d", rt_num,
		  fma_mac_to_hostname(rmt_nic->mac_addr),
		  lcl_port, rmt_port);
	}
      }
    }

  /* 1 local port or num_routes is a per-NIC value */
  } else {
      
    port_index = 0;
    port_rt_num[0] = 0;
    port_rt_num[1] = 0;

    for (rt_num = 0; rt_num < lcl_nic->myri_info.num_routes; rt_num++) {
      lcl_port = lcl_ports[port_index];
      rmt_port = rmt_ports[port_index];

      rindex = FMA_ROUTE_INDEX(lcl_nic->nic_index, lcl_port, rmt_port)
	+ port_rt_num[port_index];
      rlen = FMA_NIC(rmt_nic)->route_lens[rindex];
      if (rlen == -1) {
	port_rt_num[port_index] = 0;
	rindex = FMA_ROUTE_INDEX(lcl_nic->nic_index, lcl_port, rmt_port)
	  + port_rt_num[port_index];
      }

      rc = fma_set_route(lcl_nic, lcl_port, rmt_nic, rmt_port,
			 rindex, rt_num);
      if (rc == -1) {
	fma_log("Error setting route %d to %s", rt_num,
		fma_mac_to_hostname(rmt_nic->mac_addr));
      }
      ++port_rt_num[port_index];
      ++port_index;
      if (port_index > 1) {
	port_index = 0;
      }
    }
  }
}

/*
 * set route for all NICs
 */
int
fma_set_all_routes()
{
  int nics_in_partition;
  struct lf_fabric *fp;
  struct fma_nic_info *lcl_nip;   /* this host NIC info */
  int lcl_nic_id;                 /* this hosts nic id */
  int lcl_port;                    /* local nic interface */
  struct lf_nic *rmt_nic;         /* remote nic struct */
  int rmt_nic_id;                 /* remote nic id */
  int rmt_port;                    /* remote nic interface */
  int8_t rt_0_len[LF_MAX_NIC_PORTS][LF_MAX_NIC_PORTS];
                                  /* route 0 lengths. 1st desc is local port, 
                                   * 2nd is remote port */
  int rc;

  fp = A.fabric;

  for (lcl_nic_id = 0; lcl_nic_id < A.myri->num_nics; ++lcl_nic_id) {
    lcl_nip = A.myri->nic_info[lcl_nic_id];

    /* If this NIC is not in fabric, must be disconnected */
    if (lcl_nip->nic_ptr == NULL) continue;

    /* Start setting routes */
    rc = myri_set_route_start(lcl_nip->nic_handle);
    if (rc == -1) LF_ERROR(("Error from myri_set_route_start."));

    for (rmt_nic_id = 0; rmt_nic_id < FMA_FABRIC(fp)->num_nics; ++rmt_nic_id) {

      rmt_nic = FMA_FABRIC(fp)->nics[rmt_nic_id];

      /* If remote NIC not in same partition, set no routes to it */
      if (rmt_nic->partition != lcl_nip->nic_ptr->partition) {
	if (A.debug) {
	  fma_log("%s not in my partition",
	      fma_mac_to_hostname(rmt_nic->mac_addr));
	}
	continue;
      }

      /* for singular routes, route from interface 0 to interface 0 */
      if (lcl_nip->myri_info.num_routes == 1) {
	int rindex;
	rindex = FMA_ROUTE_INDEX(lcl_nic_id, 0, 0);
        rc = fma_set_route(lcl_nip, 0, rmt_nic, 0, 0, 0);
	if (rc == -1) LF_ERROR(("Error from fma_set_route."));
        continue;
      }

      /* get route 0 lengths for all possible route combos between
       * lcl and rmt nic interfaces. 
       */
      memset(rt_0_len, -1, sizeof(rt_0_len));
      for (lcl_port = 0; lcl_port < lcl_nip->nic_ptr->num_ports; lcl_port++) {
	for (rmt_port = 0; rmt_port < rmt_nic->num_ports; rmt_port++) {
	  rt_0_len[lcl_port][rmt_port] =
		FMA_ROUTE_LEN(rmt_nic, lcl_nic_id, lcl_port, rmt_port, 0);
	}
      }

      /* See if routes between multiple interfaces on lcl and rmt nics exist.
       * XXX - assumes LF_MAX_NIC_PORTS == 2 !!!
       */
      if ((rt_0_len[0][0] >= 0 && rt_0_len[1][1] >= 0) ||
          (rt_0_len[0][1] >= 0 && rt_0_len[1][0] >= 0)) {

        /* routes between multiple interfaces exist. Decide which ports to
         * route between based which ones exist and are the shortest.
         */
        if ((rt_0_len[0][0] >= 0 && rt_0_len[1][1] >= 0) &&
            (rt_0_len[0][1] >= 0 && rt_0_len[1][0] >= 0)) {

          /* all combos exist, pick the shortest */ 
          if (rt_0_len[0][0] + rt_0_len[1][1]
              <= rt_0_len[0][1] + rt_0_len[1][0]) {
            /* Set up to use routes from 0->0 and 1->1 because they are shorter
             * than routes from 0->1 and 1->0 */
            fma_set_route_group(lcl_nip, rmt_nic, 0, 0, 1, 1);

          } else {
            /* use routes from 0->1 and 1->0 because they are shorter */
            fma_set_route_group(lcl_nip, rmt_nic, 0, 1, 1, 0);

          }
        } else if (rt_0_len[0][0] >= 0 && rt_0_len[1][1] >= 0) {
          /* 0->0 and 1->1 exist, so use them */
          fma_set_route_group(lcl_nip, rmt_nic, 0, 0, 1, 1);

        } else if (rt_0_len[0][1] >= 0 && rt_0_len[1][0] >= 0) {
          /* 0->1 and 1->0 exist, so use them */
          fma_set_route_group(lcl_nip, rmt_nic, 0, 1, 1, 0);

        } else {
          /* this can't happen */
          LF_ERROR(("Multiple interface routing confusion"));

        }

      } else if (rt_0_len[0][0] >= 0 && rt_0_len[0][1] >= 0) {
        /* Use routes from lcl port 0 to rmt port 0 and 1 */
        fma_set_route_group(lcl_nip, rmt_nic, 0, 0, 0, 1);

      } else if (rt_0_len[1][0] >= 0 && rt_0_len[1][1] >= 0) {
        /* Use routes from lcl port 1 to rmt port 0 and 1 */
        fma_set_route_group(lcl_nip, rmt_nic, 1, 0, 1, 1);

      } else if (rt_0_len[0][0] >= 0 && rt_0_len[1][0] >= 0) {
        /* Use routes from lcl port 0 and 1 to rmt port 0 */
        fma_set_route_group(lcl_nip, rmt_nic, 0, 0, 1, 0);

      } else if (rt_0_len[0][1] >= 0 && rt_0_len[1][1] >= 0) {
        /* Use routes from lcl port 0 and 1 to rmt port 1 */
        fma_set_route_group(lcl_nip, rmt_nic, 0, 1, 1, 1);

      } else {
        /* Next try 1->1 routes.  Of all route groups, chose the one with
           the shortest routes.  (There may be more than one group of routes
           if some of the code above is disabled.) */
        int best_lcl_port=0, best_rmt_port=0;
        unsigned int shortest_len = FMA_IFC_ROUTE_LEN + 1;

        /* Find the group with the shortest routes. */
        for (lcl_port = 0; lcl_port < 2; lcl_port++) {
          for (rmt_port = 0; rmt_port < 2 ; rmt_port++) {
            if (rt_0_len[lcl_port][rmt_port] >= 0 &&
                  rt_0_len[lcl_port][rmt_port] < shortest_len) {
               shortest_len = rt_0_len[lcl_port][rmt_port];
               best_lcl_port = lcl_port;
               best_rmt_port = rmt_port;
            }
          }
        }

        /* If there is a usable group of routes, use them. */
        if (shortest_len <= FMA_IFC_ROUTE_LEN) {
          fma_set_route_group(lcl_nip, rmt_nic, best_lcl_port, best_rmt_port,
                best_lcl_port, best_rmt_port);
	}
      }
    }

    /* Count the number of NICs in my partition */
    nics_in_partition = 0;
    for (rmt_nic_id = 0; rmt_nic_id < FMA_FABRIC(fp)->num_nics; ++rmt_nic_id) {

      rmt_nic = FMA_FABRIC(fp)->nics[rmt_nic_id];
      if (rmt_nic->partition == lcl_nip->nic_ptr->partition) {
	++nics_in_partition;
      }
    }

    /* Let the API know we are done for this NIC */
    rc = myri_set_route_end(lcl_nip->nic_handle,
			    A.map_info->mi_mapper_mac_addr,
			    A.map_info->mi_map_version,
			    nics_in_partition);
    if (rc == -1) LF_ERROR(("Error from myri_set_route_end."));

  }
  return 0;

 except:
  fma_perror();
  return -1;
}

/*
 * set a route from one NIC/port to another. fma_set_route() must be
 * bracketed by calls to myri_set_route_start() and myri_set_route_end() 
 */
int
fma_set_route(
  struct fma_nic_info *nip,
  int local_port,
  struct lf_nic *nicp2,
  int remote_port,
  int rindex,
  int route_num)
{
  int rc;
  int rlen;
  unsigned char *route;

  /* get the route and length */
  rlen = FMA_NIC(nicp2)->route_lens[rindex];
  route = FMA_NIC(nicp2)->route_buf + FMA_ROUTE_OFFSET(rindex);

  /*
   * Save the first route to this NIC, just so we have one.
   * This may get done a couple of times per host (once per NIC) but
   * keeping the last one is easier than keeping the first one and
   * it doesn't matter which one we send to anyhow.
   */
  if (route_num == 0) {
    struct fma_host *fhp;

    fhp = FMA_HOST(nicp2->host);
    fhp->quick_route_len = rlen;
    if (rlen >= 0) {
      memcpy(fhp->quick_route, route, rlen);
      fhp->quick_nic = nip;
      fhp->quick_port = local_port;
    }
  }

  /* If no route, skip it */
  if (rlen < 0) {
    return 0;
  }

  rc = myri_set_route(nip->nic_handle,
  	nicp2->mac_addr,
	FMA_NIC(nicp2)->fw_type,
	remote_port,
	route_num,
	local_port,
	route, rlen);
  return rc;
}
/*
 * Figure out how far every xbar is from this host
 */
void
fma_calc_xbar_dists()
{
  int n;
  int port;
  struct lf_xbar *work_list;
  struct lf_xbar *new_list;
  struct lf_fabric *fp;

  fp = A.fabric;

  /* Initialize all xbars to have unknown distance */
  for (n=0; n<fp->num_xbars; ++n) {
    struct fma_xbar *fxp;

    fxp = FMA_XBAR(fp->xbars[n]);
    fxp->host_dist = -1;
    memset(fxp->xbar_dist, -1, A.myri->nic_ports);
  }

  /*
   * Calculate a distance for each xbar from each port on each NIC.
   * Also keep track of the closest way to get to each xbar.
   */
  for (n=0; n<A.myri->num_nics; ++n) {
    struct fma_nic_info *nip;
    struct lf_nic *nicp;

    nip = A.myri->nic_info[n];
    nicp = nip->nic_ptr;

    /* If this NIC not in map, then no lf_nic struct for it */
    if (nicp == NULL) continue;
    
    /* start with xbar attached to each port on each NIC */
    for (port=0; port < nicp->num_ports; ++port) {
      union lf_node *np;
      struct fma_xbar *fxp;
      int index;

      /* index into xbar arrays */
      index = FMA_NICPORT_INDEX(n, port);

      /* If this link is down, skip it */
      if (nicp->link_state[port] != LF_LINK_STATE_UP) continue;

      /* get connection and make sure it's an xbar */
      np = nicp->topo_ports[port];
      if (np == NULL) continue;
      if (np->ln_type != LF_NODE_XBAR && np->ln_type != LF_NODE_NIC) {
	LF_ERROR(("Non xbar/NIC attached to NIC!"));
      }
      if (np->ln_type != LF_NODE_XBAR) continue;

      /* Put this xbar alone on the work list */
      fxp = FMA_XBAR_N(np);

      fxp->bf_next = NULL;
      work_list = LF_XBAR(np);

      fxp->xbar_dist[index] = 0;

      /* maintain temp route to each xbar */
      fxp->fx_route_len = 0;
      fxp->fx_in_port = nicp->topo_rports[port];

      /* set host distance if this is closest so far.  distribute 
       * xbars of equal distances among various NICs randomly
       */
      if (fxp->host_dist == -1 ||
	  fxp->xbar_dist[index] < fxp->host_dist ||
	  (fxp->xbar_dist[index] == fxp->host_dist &&
	   (random()%A.myri->nic_ports) <= (A.myri->nic_ports-index-1))) {
	fxp->host_dist = 0;
	fxp->host_nic = nip;
	fxp->host_nic_port = port;
	fxp->host_in_port = nicp->topo_rports[port];
      }

      /* Work the list BFS */
      new_list = NULL;

      while (work_list != NULL) {
	struct lf_xbar *xp;
	int p;

	/* take the top item off the work list */
	xp = work_list;
	work_list = FMA_XBAR(xp)->bf_next;

	/* Check each port for an xbar to add */
	for (p=0; p<xp->num_ports; ++p) {
	  union lf_node *np;
	  struct fma_xbar *fxp;
	  struct fma_xbar *nfxp;

	  fxp = FMA_XBAR(xp);

	  /* If this link is down, skip it */
	  if (xp->link_state[p] != LF_LINK_STATE_UP) continue;

	  /* If this is an xbar not yet distanced, process it! */
	  np = xp->topo_ports[p];
	  if (np != NULL
	      && np->ln_type == LF_NODE_XBAR
	      && FMA_XBAR_N(np)->xbar_dist[index] == -1) {

	    /* Add this xbar to the work list */
	    nfxp = FMA_XBAR_N(np);
	    nfxp->bf_next = new_list;
	    new_list = LF_XBAR(np);

	    /* and set its host distance values */
	    nfxp->xbar_dist[index] = fxp->xbar_dist[index] + 1;
 
	    /* maintain route info to each xbar */
	    memcpy(nfxp->fx_route, fxp->fx_route, fxp->fx_route_len);
	    nfxp->fx_route[nfxp->xbar_dist[index]-1] = 
	      LF_DELTA_TO_ROUTE(p - fxp->fx_in_port);
	    nfxp->fx_route_len = fxp->fx_route_len + 1;
	    nfxp->fx_in_port = xp->topo_rports[p];

	    /* see if we have found a closer way to this xbar */
	    if (nfxp->host_dist == -1 ||
		nfxp->xbar_dist[index] < nfxp->host_dist ||
		(nfxp->xbar_dist[index] == nfxp->host_dist &&
		 (random()%A.myri->nic_ports) <= (A.myri->nic_ports-index-1))) {

	      nfxp->host_dist = nfxp->xbar_dist[index];
	      nfxp->host_nic = nip;
	      nfxp->host_nic_port = port;
	      nfxp->host_in_port = nfxp->fx_in_port;

	      memcpy(nfxp->host_route, nfxp->fx_route, nfxp->host_dist);
	    }
	  }
	}

	/* If the list is empty, switch to new list */
	if (work_list == NULL) {
	  work_list = new_list;
	  new_list = NULL;
	}
      }
    }
  }
  return;

 except:
  fma_perror_exit(1);
}

/*
 * Check all quick routes to make sure they are valid.
 * Mark each one that is invalid and return the number that are invalid.
 */
int
fma_check_quick_routes()
{
  struct lf_fabric *fp;
  int num_bad;
  int h;
  
  fp = A.fabric;
  num_bad = 0;

  for (h=0; h<fp->num_hosts; ++h) {
    struct lf_host *hp;
    struct lf_nic *my_nicp;
    int my_port;
    union lf_node *dest_np;
    int dest_port;

    hp = fp->hosts[h];

    /* skip self */
    if (hp == A.my_host) continue;

    /* check here for obvious badness */
    if (FMA_HOST(hp)->quick_route_len == -1
    	|| FMA_HOST(hp)->quick_nic == NULL) {
      ++num_bad;
      FMA_HOST(hp)->quick_route_len = -1;	/* mark it bad */
      continue;
    }

    /* follow the quick route */
    my_nicp = FMA_HOST(hp)->quick_nic->nic_ptr;

    /* more obvious badness */
    if (my_nicp == NULL) {
      ++num_bad;
      FMA_HOST(hp)->quick_route_len = -1;	/* mark it bad */
      continue;
    }

    my_port = FMA_HOST(hp)->quick_port;
    dest_np = lf_follow_route(LF_NODE(my_nicp), my_port,
		      FMA_HOST(hp)->quick_route, FMA_HOST(hp)->quick_route_len,
		      &dest_port);

    /* Check where we landed */
    if (dest_np == NULL
	|| dest_np->ln_type != LF_NODE_NIC
	|| LF_NIC(dest_np)->host != hp) {
      ++num_bad;
      FMA_HOST(hp)->quick_route_len = -1;	/* mark it bad */
    }
  }

  return num_bad;
}
